## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.path   = "figure-html/",
  fig.width  = 12,
  fig.height = 8 
)

## ----eval = FALSE-------------------------------------------------------------
# install.packages("shadowVIMP")

## ----setup, warning = FALSE, message = FALSE----------------------------------
library(shadowVIMP)
library(magrittr)
library(dplyr)
library(ggplot2)

## ----ex_default---------------------------------------------------------------
data("mtcars")
set.seed(786)

is_interactive <- interactive()
if (is_interactive) {
  # When interactive - set num_cores to a selected value
  global_num_threads <- min(parallel::detectCores()/2, 6)
} else{
  # Value of num.threads parameter for CRAN 
  global_num_threads <- 1
}

# WARNING 1: The specified values of the niters parameter are too small! To get reliable results, use the default or higher values of the niters parameter.
# WARNING 2: To avoid potential issues with parallel computing on CRAN, we set num.threads to 1, by default it is set to half of the available CPU cores, which speeds up computation.
vimp_def <- shadow_vimp(data = mtcars, outcome_var = "vs", niters = c(20, 30, 50),
                        num.threads = global_num_threads)
vimp_def

## ----ex_default2--------------------------------------------------------------
vimp_def$final_dec_pooled %>%
  head()

## ----plot_base_starts, eval=TRUE----------------------------------------------
plot_vimps(shadow_vimp_out = vimp_def)

## ----echo=FALSE, out.width="85%", fig.cap="Permutation scheme"----------------
knitr::include_graphics("rshadow.png")

## ----echo=FALSE, out.width="85%", fig.cap="Illustration of pre-selection procedure"----
knitr::include_graphics("preselection.png")

## ----inspect_history1---------------------------------------------------------
# Reminder - definition of vimp_def object:
# set.seed(786)
# vimp_def <- shadow_vimp(data = mtcars, outcome_var = "vs", niters = c(20, 30, 50),
#                         num.threads = global_num_threads)

# VIMP history from the 1st step for 5 covariates:
vimp_def$pre_selection$step_1$vimp_history %>%
  select(1:5) %>%
  head()

# VIMP history from the 2nd step for 5 covariates:
vimp_def$pre_selection$step_2$vimp_history %>%
  select(1:5) %>%
  head()

## ----vimp_last_step-----------------------------------------------------------
vimp_def$vimp_history %>%
  select(1:5) %>%
  head()

## ----fdr_unadjusted_dec-------------------------------------------------------
# Show FDR and unadjusted p-values
vimp_fdr <- shadow_vimp(data = mtcars, outcome_var = "vs",
                        to_show = "FDR", niters = c(20, 30, 50),
                        num.threads = global_num_threads)
vimp_fdr

# Show only unadjusted p-values
vimp_unadjusted <- shadow_vimp(data = mtcars, outcome_var = "vs",
                               to_show = "unadjusted", niters = c(20, 30, 50),
                               num.threads = global_num_threads)
vimp_unadjusted

## ----per_variable-------------------------------------------------------------
vimp_per_variable <- shadow_vimp(data = mtcars, outcome_var = "vs",
                                 method = "per_variable", niters = c(20, 30, 50),
                                 num.threads = global_num_threads)

vimp_per_variable$final_dec_per_variable %>%
  head()

## ----reminder-----------------------------------------------------------------
# Definition of vimp_def:
# vimp_def <- shadow_vimp(data = mtcars, outcome_var = "vs", niters = c(20, 30, 50),
#                         num.threads = global_num_threads)
vimp_def

## ----time---------------------------------------------------------------------
vimp_def$time_elapsed

## ----minor--------------------------------------------------------------------
vimp_def$alpha
vimp_def$step_all_covariates_removed

## ----pre_selection------------------------------------------------------------
# VIMP from 1st step
vimp_def$pre_selection$step_1$vimp_history %>%
  select(1:5) %>%
  head()

# Which covariates were considered as informative in the 1st step?
vimp_def$pre_selection$step_1$decision_pooled %>%
  head()

# The significance level used in the 1st step
vimp_def$pre_selection$step_1$alpha

## ----plot_base----------------------------------------------------------------
plot_vimps(shadow_vimp_out = vimp_def)

## ----plot_txt_size,  eval=TRUE------------------------------------------------
plot_vimps(shadow_vimp_out = vimp_def, text_size = 5)

## ----filter_vars,  eval=TRUE--------------------------------------------------
plot_vimps(shadow_vimp_out = vimp_def, filter_vars = 4)

## ----legend_position, eval=TRUE-----------------------------------------------
# Legend on the bottom
plot_vimps(shadow_vimp_out = vimp_def, legend.position = "bottom", text_size = 3)

# Plot without the legend
plot_vimps(shadow_vimp_out = vimp_def, legend.position = "none", text_size = 3)

## ----p_val_labels,  eval=TRUE-------------------------------------------------
# No p-values labels
plot_vimps(shadow_vimp_out = vimp_def, p_val_labels = FALSE)

## ----no_helper_plot,  eval=TRUE-----------------------------------------------
plot_vimps(shadow_vimp_out = vimp_def, helper_legend = FALSE, , text_size = 3)

## ----change_colors,  eval=TRUE------------------------------------------------
plot_vimps(
  shadow_vimp_out = vimp_def,
  category_colors = c(
    "FWER conf." = "#F56455FF",
    "FDR conf." = "#15134BFF",
    "Unadjusted conf." = "#87C785FF",
    "Not significant" = "#572F30FF"
  ),
  text_size = 3
)

## ----other_options,  eval=TRUE------------------------------------------------
plot_vimps(shadow_vimp_out = vimp_def, text_size = 3) +
  patchwork::plot_annotation(
    title = "My Cool Plot",
    subtitle = "Even better subtitle"
  ) &
  theme(plot.title = element_text(size = 16, face = "bold"))

## ----plot_per_var, eval=FALSE-------------------------------------------------
# vimp_per_variable <- shadow_vimp(data = mtcars, outcome_var = "vs",
#                                  method = "per_variable", niters = c(20, 30, 50),
#                                  num.threads = global_num_threads)

## ----plot_per_var2, eval=TRUE-------------------------------------------------
plot_vimps(shadow_vimp_out = vimp_per_variable, pooled = FALSE, text_size = 3)

## ----parallel1, echo = TRUE, eval = TRUE--------------------------------------
# Detect if running on non-interactive environments
is_interactive <- interactive()
if (is_interactive) {
  # When interactive - set num_cores to a selected value
  # Here we want to use 6 cores or if 6 cores are not available, then the number of available cores
  num_cores <- min(parallel::detectCores(), 6)
  vimp_parallel1 <- shadow_vimp(data = mtcars, outcome_var = "vs", niters = c(20, 30, 50),
                        num.threads = num_cores)
} else {
  # When non-interactive - run with 1 thread
  vimp_parallel1 <- shadow_vimp(data = mtcars, outcome_var = "vs", niters = c(20, 30, 50),
                        num.threads = 1)
}

